import 'package:flutter/material.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:flutter_screenutil/flutter_screenutil.dart';
import 'package:gap/gap.dart';
import 'package:ionicons/ionicons.dart';
import 'package:razinsoft_rider/common/loading_view.dart';
import 'package:razinsoft_rider/core/enums/car_view_type.dart';
import 'package:razinsoft_rider/core/extensions/extensions.dart';
import 'package:razinsoft_rider/core/utils/helpers.dart';
import 'package:razinsoft_rider/core/widgets/is_ios.dart';
import 'package:razinsoft_rider/generated/l10n.dart';
import 'package:razinsoft_rider/presentation/account_page/provider/theme_provider.dart';
import 'package:razinsoft_rider/presentation/booking/provider/order_providers.dart';
import 'package:razinsoft_rider/presentation/booking/widgets/apply_coupon_bottom_sheet.dart';
import 'package:razinsoft_rider/presentation/booking/widgets/service_list.dart';
import 'package:razinsoft_rider/presentation/dashboard/viewmodel/car_type_notifier.dart';
import 'package:razinsoft_rider/presentation/waypoint/widgets/app_card_sheet.dart';

import '../../../core/widgets/buttons/app_primary_button.dart';
import '../../../core/widgets/buttons/app_text_button.dart';
import '../../../gen/assets.gen.dart';
import '../provider/ride_services_providers.dart';
import '../provider/selection_providers.dart';
import 'ride_preferences_sheet.dart';

class RideBookingSheet extends ConsumerWidget {
  const RideBookingSheet({super.key});

  @override
  Widget build(BuildContext context, WidgetRef ref) {
    final riderServiceState = ref.watch(rideServicesNotifierProvider);
    final isGridView = ref.watch(carTypeNotifierProvider).viewType == CarViewType.grid;
    final isDark = ref.watch(themeModeProvider.notifier).isDarkMode();

    return AppCardSheet(
      child: SafeArea(
        top: false,
        bottom: !isIos(),
        child: Column(
          mainAxisSize: MainAxisSize.min,
          children: [
            _buildHeader(context, isDark: isDark),
            Gap(16.h),
            _buildPreferenceRow(context, isDark: isDark),
            _buildServiceList(context, riderServiceState, isGridView, ref),
            _buildBookNowButton(isDark: isDark),
          ],
        ),
      ),
    );
  }

  Widget _buildHeader(BuildContext context, {required bool isDark}) => Text(
      AppLocalizations.of(context).choose_ride_title,
      textAlign: TextAlign.center,
      style: context.bodyLarge?.copyWith(
        fontSize: 20.sp,
        fontWeight: FontWeight.w600,
        color: isDark ? Colors.white : Colors.black,
      ),
    );

  Widget _buildPreferenceRow(BuildContext context,{required bool isDark}) => Row(
      children: [
        AppTextButton(
          isDense: true,
          text: AppLocalizations.of(context).ride_preferences,
          iconData: Ionicons.options,
          onPressed: () async {
            await showModalBottomSheet(
              context: context,
              enableDrag: false,
              isScrollControlled: true,
              isDismissible: false,
              builder: (_) => RidePreferencesSheet(isDark: isDark,),
            );
          }, isDark: isDark,
        ),
        const Spacer(),
        _buildCouponButton(context),
        _buildListGridToggle(),
      ],
    );

  Widget _buildCouponButton(BuildContext context) => InkWell(
      onTap: () async {
        await showModalBottomSheet(
          context: context,
          isScrollControlled: true,
          builder: (_) => const ApplyCouponBottomSheet(),
        );
      },
      child: Assets.images.coupon.image(
        height: 40.h,
        width: 40.w,
        fit: BoxFit.fill,
      ),
    );

  Widget _buildListGridToggle() => Consumer(
      builder: (context, ref, _) {
        final carType = ref.watch(carTypeNotifierProvider);
        final notifier = ref.read(carTypeNotifierProvider.notifier);
        final icon = carType.viewType == CarViewType.grid
            ? Assets.images.list
            : Assets.images.grid;

        return InkWell(
          onTap: () => notifier.toggleViewType(),
          child: icon.image(height: 40.h, width: 40.w, fit: BoxFit.fill),
        );
      },
    );

  Widget _buildServiceList(BuildContext context, dynamic state, bool isGridView, WidgetRef ref) => SizedBox(
      height: isGridView ? 200 : 146,
      width: double.infinity,
      child: Builder(
        builder: (_) => state.when(
          initial: () => _refreshWrapper(ref, Center(child: Text(AppLocalizations.of(context).initializing))),
          loading: () => _refreshWrapper(ref, const Center(child: LoadingView())),
          error: (e) => _refreshWrapper(ref, Center(child: Text(e.message))),
          success: (data) {
            final list = data.data?.servicesList ?? [];
            if (list.isEmpty) {
              return _refreshWrapper(ref, Center(child: Text(AppLocalizations.of(context).no_service_available)));
            }
            return ServiceList(data: data);
          },
        ),
      ),
    );

  Widget _refreshWrapper(WidgetRef ref, Widget child) => RefreshIndicator(
      onRefresh: () async => ref.invalidate(rideServicesNotifierProvider),
      child: ListView(
        physics: const AlwaysScrollableScrollPhysics(),
        children: [SizedBox(height: 200, child: child)],
      ),
    );

  Widget _buildBookNowButton({required bool isDark}) => Consumer(
      builder: (context, ref, _) {
        final notifier = ref.read(createOrderNotifierProvider.notifier);

        Future<void> handleOrderCreation() async {

          final selectedService = ref.read(carTypeNotifierProvider);
          final bookingData = ref.read(rideServiceFilterNotiferProvider);

          if(selectedService.selectedCarType == null){
            showNotification(message: AppLocalizations.of(context).select_service);
            return;
          }
          final data = {
            'pickup_location': [bookingData.pickupLocation.first, bookingData.pickupLocation.last],
            'drop_location': [bookingData.dropLocation.first, bookingData.dropLocation.last],
            'wait_location': bookingData.waitLocation.isNotEmpty
                ? [bookingData.waitLocation.first, bookingData.waitLocation.last]
                : [],
            'service_option_ids': bookingData.serviceOptionIds,
            'coupon_code': bookingData.couponCode,
            'pickup_address': bookingData.pickupAddress,
            'drop_address': bookingData.dropAddress,
            'wait_address': bookingData.waitAddress,
            'service_id': selectedService.selectedCarType?.id,
          };

          await notifier.createOrder(orderData: data);
        }

        final bookNowBtn = Padding(
          padding: EdgeInsets.only(top: 16.h),
          child: AppPrimaryButton(
            isLoading: ref.watch(createOrderNotifierProvider).whenOrNull(loading: ()=> true) ?? false,
            onPressed: handleOrderCreation,
            child: Text(AppLocalizations.of(context).start_ride, style: context.bodyMedium?.copyWith(color: Colors.white),),
          ),
        );

        return bookNowBtn;
      },
    );
}
